; Motor speed controller: February 2014

	ERRORLEVEL -302
	ERRORLEVEL -306

	list P=16F88
	#include p16f88.inc

;Program Configuration Register 1
		__CONFIG    _CONFIG1, _CP_OFF & _CCP1_RB3  & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _LVP_OFF & _BODEN_OFF & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _INTRC_IO

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF

; Define variables at memory locations

; Bank 0 RAM

VR2_3		equ	H'20'	; ADCON0 value for AN2 or AN3	
SPEED_SET	equ	H'21'	; speed setting of VR1
GAIN		equ	H'22'	; feedback gain set by VR2/3
CURRENT		equ	H'23'	; current as read at AN4 
FEEDBACK	equ	H'24'	; PWM change due to feedback
START_FLG	equ	H'25'	; startup flag
REQ_PWM		equ	H'26'	; required PWM
FEED_PWM	equ	H'27'	; feedback PWM
STORE1		equ	H'28'	; delay store
STORE2		equ	H'29'	; delay store  
DELAY_VAL	equ	H'2A'	; delay value
PERIOD		equ	H'2B'	; 10ms period flag
CURRENT0	equ	H'2C'	; total of current (ms byte)
CURRENT1	equ	H'2D'	; total of current
NUMBER		equ	H'2E'	; number of current counts 
OVER_I		equ	H'2F'	; Over current flag
RUN_I		equ	H'30'	; no load idle or running current
TEMP_GAIN	equ	H'31'	; temporary gain setting storage

; math routines
TEMP1		equ	H'61'
TEMPB1      equ H'62'
TEMPB0      equ H'63'
TEMPD       equ H'64'    ; temporary storage
AARGB3		equ	H'65'
AARGB2      equ H'66'
AARGB1      equ H'67'
AARGB0      equ H'68'	; most significant byte of argument A
    
BARGB1      equ H'69'
BARGB0      equ H'6A'	; most significant byte of argument B
    
REMB1       equ H'6B'
REMB0       equ H'6C'   ; most significant byte of remainder
LOOPCOUNT   equ H'6D'   ; loop counter
STO1		equ	H'6E'	; storage

; All Banks RAM
; Interrupt store registers 
W_TMP			equ	H'72'	; storage of w before interrupt
STATUS_TMP		equ	H'73'	; status storage before interrupt


	org	0	
	goto	SETUP
	nop
	nop
	nop
	org	4	
	goto	INTERRUPT


; *********************************************
; feedback pot non-linearisation part of from A/D readings from 192 to 255
NON_LIN
	addwf 	PCL,f			; lookup table for feedback control

; for VR2/VR3 A/D readings 192 to 255 

	retlw	D'129'			; 192
	retlw	D'131'			; 193 
	retlw	D'133'			; 194 
	retlw	D'135'			; 195
	retlw	D'137'			; 196
	retlw	D'139'			; 197
	retlw	D'141'			; 198
	retlw	D'143'			; 199  
	retlw	D'145'			; 200
	retlw	D'147'			; 201
	retlw	D'149'			; 202
	retlw	D'151'			; 203 
	retlw	D'153'			; 204 
	retlw	D'155'			; 205
	retlw	D'157'			; 206
	retlw	D'159'			; 207
	retlw	D'161'			; 208
	retlw	D'163'			; 209  
	retlw	D'165'			; 210
	retlw	D'167'			; 211
	retlw	D'169'			; 212
	retlw	D'171'			; 213 
	retlw	D'173'			; 214 
	retlw	D'175'			; 215
	retlw	D'177'			; 216
	retlw	D'179'			; 217
	retlw	D'181'			; 218
	retlw	D'183'			; 219  
	retlw	D'185'			; 220
	retlw	D'187'			; 221
	retlw	D'189'			; 222
	retlw	D'191'			; 223 
	retlw	D'193'			; 224 
	retlw	D'195'			; 225
	retlw	D'197'			; 226
	retlw	D'199'			; 227
	retlw	D'201'			; 228
	retlw	D'203'			; 229  
	retlw	D'205'			; 230
	retlw	D'207'			; 231
	retlw	D'209'			; 232
	retlw	D'211'			; 233 
	retlw	D'213'			; 234 
	retlw	D'215'			; 235
	retlw	D'217'			; 236
	retlw	D'219'			; 237
	retlw	D'221'			; 238
	retlw	D'223'			; 239  
	retlw	D'225'			; 240
	retlw	D'227'			; 241
	retlw	D'229'			; 242
	retlw	D'231'			; 243 
	retlw	D'233'			; 244 
	retlw	D'235'			; 245
	retlw	D'237'			; 246
	retlw	D'239'			; 247
	retlw	D'241'			; 248
	retlw	D'243'			; 249  
	retlw	D'245'			; 250
	retlw	D'247'			; 251
	retlw	D'249'			; 252
	retlw	D'251'			; 253 
	retlw	D'253'			; 254 
	retlw	D'255'			; 255
	retlw	D'255'			; 256

; *********************************************
; INTERRUPT

; start interrupt by saving w and status registers 
	
INTERRUPT
	movwf	W_TMP			; w to w_tmp storage
	swapf	STATUS,w		; status to w
	movwf	STATUS_TMP		; status in status_tmp 
	bcf		STATUS,RP0		; bank select
	bcf 	STATUS,RP1		; select memory bank 0 

	btfss	PIR1,TMR1IF		; timer 1 interrupt flag
	goto	CHECK_END		; no timer interrupt so exit
	bcf		PIR1,TMR1IF		; clear flag
	
	bsf		PERIOD,0

; preset timer 1 for 10ms
	movlw	H'D8'
	movwf	TMR1H			; timer high byte
	movlw	H'F0'			; 
	movwf	TMR1L

; end of interrupt reclaim w and status 
CHECK_END
	swapf	STATUS_TMP,w; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie				; return from interrupt
; End of Interrupt

; *************************************************
SETUP	
	movlw	H'00'		; set all outputs low
	movwf	PORTA
	movwf	PORTB
	clrf	START_FLG	; startup flag
	clrf	REQ_PWM		; required PWM
	clrf	RUN_I		; no load running current

; set inputs/outputs
	
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000111'	; comparators off
	movwf	CMCON
	movlw	B'00000001'	; I/O (RB outputs)
	movwf	TRISB		; port B data direction register
	movlw	B'11000000'	; 
	movwf	OPTION_REG	; pullups disabled and interupt on rising edge for RB0
	movlw   B'00111110'	; I/O 
	movwf   TRISA		; port A data direction register
; analog inputs, A/D
	movlw	B'00011110'	; AN1-AN4 analog input 
	movwf	ANSEL
	movlw	B'00000000'	; left justified A/D result, Vdd to Vss A/D
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'01001000'	; Fosc, channel 2  etc
	movwf	ADCON0
; oscillator
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'01100000'	; for 4MHz
	movwf	OSCCON		; osc

	bcf		STATUS,RP0	; select memory bank 0

; preset timer 1 for 10ms
	movlw	H'D8'
	movwf	TMR1H			; timer high byte
	movlw	H'F0'			; 
	movwf	TMR1L	
	movlw	B'00000000'	; timer 1 fosc/4
	movwf	T1CON

	bsf		STATUS,RP0	; select memory bank 1	

; pwm set
	movlw	D'255'		; 976Hz; D'199'		; for 1.25kHz
	movwf	PR2			; PWM period register
	bcf		STATUS,RP0	; memory bank 0
	clrf	CCPR1L		; ms byte of PWM
	movlw	B'00000001'	; /4
	movwf	T2CON
	bsf		T2CON,2		; enable timer 2
	movlw	B'00001100'	; set PWM mode
	movwf	CCP1CON		; enable PWM operation

; AN2 or AN3 is feedback, VR2/VR3. Sets feedback gain. 
; Check if VR2 is connected at startup. This is the dominant control, but if not installed, VR3 is used.
; 
; set AN2 as an output
	bsf		STATUS,RP0	; select memory bank 1
	movlw   B'00111010'	; I/O 
	movwf   TRISA		; port A data direction register
	movlw	B'00011110'	; AN1-AN4 analog input 
	movwf	ANSEL
	bcf		STATUS,RP0	; select memory bank 0
; set output high
	bsf		PORTA,2
	call	DELAY1
; set as input
	bsf		STATUS,RP0	; select memory bank 1
	movlw   B'00111110'	; I/O 
	movwf   TRISA		; port A data direction register
	movlw	B'00011110'	; AN1-AN4 analog input 
	movwf	ANSEL
	bcf		STATUS,RP0	; select memory bank 0

; check if still high
	movlw	B'01010000'	; Fosc, channel 2  etc
	movwf	ADCON0
	call	ACQUIRE_AD	; result in w
	sublw	H'F0'
	btfsc	STATUS,C	; if minus input is still high
	goto	USE_AN2		; result differs so VR2 connected

; set AN2 as an output
	bsf		STATUS,RP0	; select memory bank 1
	movlw   B'00111010'	; I/O 
	movwf   TRISA		; port A data direction register
	movlw	B'00011110'	; AN1-AN4 analog input 
	movwf	ANSEL
	bcf		STATUS,RP0	; select memory bank 0
; set output low
	bcf		PORTA,2
	call	DELAY1		
; set as input
	bsf		STATUS,RP0	; select memory bank 1
	movlw   B'00111110'	; I/O 
	movwf   TRISA		; port A data direction register
	movlw	B'00011110'	; AN1-AN4 analog input 
	movwf	ANSEL
	bcf		STATUS,RP0	; select memory bank 0

; check if still low
	movlw	B'01010000'	; Fosc, channel 2  etc
	movwf	ADCON0
	call	ACQUIRE_AD	; result in w
	sublw	H'F'
	btfss	STATUS,C	; if plus, input is still low
	goto	USE_AN2		; result differs so VR2 connected
USE_AN3	 
; AN3 is alternative feedback, VR3
; set AN2 as an output
	bsf		STATUS,RP0	; select memory bank 1
	movlw   B'00111010'	; I/O 
	movwf   TRISA		; port A data direction register
	bcf		STATUS,RP0	; select memory bank 0
; use VR3
	movlw	B'01011000'	; Fosc, channel 3  etc
	goto	STO_ADCON

USE_AN2
	movlw	B'01010000'	; Fosc, channel 2  etc
STO_ADCON
	movwf	VR2_3		; ADCON0 channel value stored

; initial conditions

	clrf	CURRENT0	; total of current (ms byte)
	clrf	CURRENT1	; total of current
	clrf	NUMBER		; number of current counts 
	clrf	OVER_I		; overcurrent flag

; allow interrupts
	bsf		STATUS,RP0		; select memory bank 1
	bsf		PIE1,TMR1IE		; timer 1 overflow interrupt
	bcf		STATUS,RP0		; select memory bank 0
	bcf		PIR1,TMR1IF		; timer 1 interrupt flag
	bsf		INTCON,PEIE		; enable periperal interrupts 	 
	bsf		INTCON,GIE

; start Timer1
	movlw	B'00000001'		; timer 1 fosc/4
	movwf	T1CON

; ****************************************************************
; normal code runs here
CYCLE
; read AN4 for current reading	
	movlw	B'01100000'	; Fosc, channel 4  etc
	movwf	ADCON0
	call	ACQUIRE_AD	; result in w

	addwf	CURRENT1,f	; add current
	btfss	STATUS,C	; if overrange increase ms byte
	goto	NXT_CALC
	movlw	D'1'
	addwf	CURRENT0,f

NXT_CALC
	incf	NUMBER,f

; check if period flag is set (10ms)
	btfss	PERIOD,0	; set in 10ms interrupt
	goto	CYCLE
	clrf	PERIOD

; during startup from zero, soft start and ignore feedback
; reset start flag when VR1 is zero
; reset start flag when current is zero
; feedback gives speed control

; 1. monitor VR1 
	movlw	B'01001000'	; Fosc, channel 1  etc
	movwf	ADCON0
	call	ACQUIRE_AD	; result in w
; read VR1
	movwf	SPEED_SET

; check if near to 0, reset start flag

	movf	SPEED_SET,w
	sublw	D'2'
	btfsc	STATUS,C
	clrf	START_FLG

; when reading feedback control use VR2_3 to load ADCON0
	movf	VR2_3,w		; VR2 or VR3
	movwf	ADCON0
; read VR2/3
	call	ACQUIRE_AD	; result in w
	movwf	TEMP_GAIN	; actual A/D reading temporary gain store

; non-linearise pot control
; if from 0-127 divide by 2
; check for <128
	movlw	D'128'
	subwf	TEMP_GAIN,w
; if negative A/D < 127
	btfsc	STATUS,C
	goto	NEXT_RANGE1
	bcf		STATUS,C
	rrf		TEMP_GAIN,w	; divide by 2 for 0-63
	goto	SET_GAIN		

NEXT_RANGE1
; if from 128 to 191 take 64 away
; check for <192
	movlw	D'192'
	subwf	TEMP_GAIN,w
; if negative is A/D < 191
	btfsc	STATUS,C
	goto	NEXT_RANGE2
	movlw	D'64'
	subwf	TEMP_GAIN,w	; subtract 64 for a range of 64 to 127
	goto	SET_GAIN

NEXT_RANGE2
; if 192 to 255 use lookup table
; take 192 away for 0-63 range (already done above in NEXT_RANGE1
	call	NON_LIN		; lookup table		
	
SET_GAIN
	movwf	GAIN		; VR2/VR3 setting for gain

; take average of current 
; divide CURRENT0,CURRENT1 by NUMBER
; Numerator
	movf	CURRENT0,w	; ms byte
	movwf	AARGB0
	movf	CURRENT1,w	; ls byte
	movwf	AARGB1
; denominator
	movf	NUMBER,w
	movwf	BARGB0
; divide
	call	DIV16_8
	movf	AARGB1,w	; result
	movwf	CURRENT		; store in CURRENT

; check when current is >15A (4.5V or D'229')
	movf	CURRENT,w
	sublw	D'245'		; use 245
	btfss	STATUS,C
	bsf		OVER_I,0	; set flag when over

; When gain is near 0 store current running current	
	movf	TEMP_GAIN,w	; genuine A/D reading of VR2/VR3
	sublw	D'2'
	btfss	STATUS,C
	goto	BYPASS_RUN_I
; if 2 or less store CURRENT to RUN_I
	movf	CURRENT,w
	movwf	RUN_I

BYPASS_RUN_I

; reset averaging values

	clrf	CURRENT0	; total of current (ms byte)
	clrf	CURRENT1	; total of current
	clrf	NUMBER		; number of current counts 

; reset start flag when current is near 0
	movf	CURRENT,w
	sublw	D'5'
	btfsc	STATUS,C
	clrf	START_FLG	; start flag

; 2. set required PWM, 
	movf	SPEED_SET,w	; VR1 setting
; compare with REQ_PWM
	xorwf	REQ_PWM,w
	btfss	STATUS,Z
	goto	CHECK_MORE_LESS
; when required PWM is equal to VR1, set START_FLG
	bsf		START_FLG,0
	goto	NEUTRAL
CHECK_MORE_LESS
; check if REQ_PWM is more or less than SPEED_SET
	movf	SPEED_SET,w	; VR1 setting
; compare with REQ_PWM
	subwf	REQ_PWM,w
	btfss	STATUS,C
	goto	INCREASE	; more REQ_PWM needed
; decrease at full rate
	movf	SPEED_SET,w
	movwf	REQ_PWM
	goto	NEUTRAL
; increase at a slow rate	
INCREASE	; more REQ_PWM needed
; check start flag if clear use 1
	movf	START_FLG,w
	movlw	D'1'		; increase rate for start up
	btfss	STATUS,Z	; check start flag	
	movlw	D'2'		; increase rate for normal running
	addwf	REQ_PWM,f
	btfss	STATUS,C	; if overflow set at FF
	goto	NEUTRAL
	movlw	H'FF'
	movwf	REQ_PWM		; set at maximum
NEUTRAL
; ignore feedback if START flag is clear
	movf	START_FLG,w
	btfsc	STATUS,Z
	goto	BY_FEED

; 3. alter REQ_PWM according to feedback at AN4 and gain setting with VR2/VR3 

; multiply GAIN by CURRENT (a 16-bit result in AARGB2 and AARGB3)
	clrf	AARGB0		; clear
	movlw	D'2'		; take 2 from gain value
	subwf	GAIN,w		; GAIN
	btfss	STATUS,C	; if negative use 0
	clrw
	movwf	AARGB1
	clrf	BARGB0

; take running (idle) current from CURRENT
	movf	RUN_I,w
	subwf	CURRENT,w
	btfss	STATUS,C	; if negative set at 0
	clrw
	movwf	CURRENT
	movf	CURRENT,w
; if zero bypass calculation
	btfsc	STATUS,Z
	goto	CURRENT_UNDER_B
	movwf	BARGB1
	call	FXM1616U	; multiply 
; result in AARGB2,3 (AARGB0,1 would be 0)
; move AARGB2,3 to AARGB0,1
	movf	AARGB2,w
	movwf	AARGB0
	movf	AARGB3,w
	movwf	AARGB1
; multiply by REQ_PWM
	clrf	BARGB0
	movf	REQ_PWM,w
	movwf	BARGB1
	call	FXM1616U	; multiply
; result in AARGB0,1,2,3 (32 bit) AARGB0 would be 0 

; shift left for divide. AARGB1 to AARGB0, AARGB2 to AARGB1, AARGB3 to AARGB2
	movf	AARGB1,w
	movwf	AARGB0
	movf	AARGB2,w
	movwf	AARGB1
	movf	AARGB3,w
	movwf	AARGB2

; divide value, 15 is good  
	movlw	D'15' 	; sets overall PWM increase with load. Smaller value = greater effect
	movwf	BARGB0
	movlw	H'00'	;
	movwf	BARGB1
	call	FXD2416U	; divide 
	movf	AARGB1,w
	movlw	D'255'
	btfsc	STATUS,Z	; if ms byte is not zero use maximum (H'FF')(D'255')
	
; result in AARGB2,3
; AARGB2 is value /255
	movf	AARGB2,w
	addwf	REQ_PWM,w	; for feedback add calculation to PWM
	btfsc	STATUS,C
	movlw	D'255'		; if overrange use maximum
	movwf	FEED_PWM 	; use FEED_PWM as new PWM value

; 4. check for over current. Reduce PWM till overcurrent is off
	btfss	OVER_I,0
	goto	CURRENT_UNDER_A
	clrf	OVER_I		; clear overcurrent flag

; overcurrent
; reduce CCPR1L till RB0 is low
CURRENT_LOOP_A
	movf	CCPR1L,w
	btfsc	STATUS,Z
	goto	CYCLE			; no PWM drive anyway
	decf	CCPR1L,f
	movf	CCPR1L,w
	movwf	FEED_PWM		; new PWM

	btfss	PORTB,0			; check for over current
	goto	CYCLE
	movlw	D'10'			; 1ms
	call 	DELAY2
	goto	CURRENT_LOOP_A

CURRENT_UNDER_A
	movf	CCPR1L,w	; current PWM
; compare with REQ_PWM
	xorwf	FEED_PWM,w
	btfsc	STATUS,Z
; when CCPR1L is equal to FEED_PWM bypass change
	goto	CYCLE
; check if FEED_PWM is more or less than CCPR1L
	movf	CCPR1L,w	; current PWM
; compare with FEED_PWM
	subwf	FEED_PWM,w
	btfsc	STATUS,C
	goto	MORE_PWM	; more PWM needed
; decrease at full rate
	movf	FEED_PWM,w
	movwf	CCPR1L
	goto	CYCLE
; increase at a slow rate	
MORE_PWM	; more REQ_PWM needed
	movlw	D'2'		; increase rate for feedback control
	addwf	CCPR1L,w
	btfsc	STATUS,C	; if overflow set at FF
	movlw	H'FF'
	movwf	CCPR1L
	goto	CYCLE

; startup bypass, ignores feedback control plus overcurent
BY_FEED ; bypass feedback
CURRENT_UNDER_B
; bypass check for over current during startup.

	clrf	OVER_I			; clear overcurrent flag
	movf	REQ_PWM,w		; new required PWM
	movwf	CCPR1L			; sets speed
	goto	CYCLE

; ****************************************************

; subroutine to wait for A/D conversion
ACQUIRE_AD
	bsf		ADCON0,ADON	; A/D on
	movlw	D'1'
	call	DELAY2
	bsf		ADCON0,2	; GO/DONE bit start conversion
WAIT_CONV
	btfsc	ADCON0,2	; conversion complete when cleared ~11 cycles
	goto	WAIT_CONV
	bcf		ADCON0,ADON	; A/D off
	movf	ADRESH,w
	return

DELAY1
	movlw	D'5'	; ~500us
DELAY2
	movwf	STORE1
LOOP2
	movlw	D'30'
	movwf	STORE2
LOOP1
	decfsz	STORE2,f
	goto	LOOP1
	decfsz	STORE1,f
	goto	LOOP2
	return

;       16/8 Bit Unsigned Fixed Point Divide 16/8 -> 16.08

;       Input:  16 bit unsigned fixed point dividend in AARGB0, AARGB1
;               8 bit unsigned fixed point divisor in BARGB0

;      ;       Output: 16 bit unsigned fixed point quotient in AARGB0, AARGB1
;               8 bit unsigned fixed point remainder in REMB0

;       Result: AARG, REM  <--  AARG / BARG

DIV16_8      	CLRF            REMB0
                MOVLW           H'08'
                MOVWF           LOOPCOUNT

LOOPU1608A      RLF             AARGB0,W
                RLF             REMB0, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F

                BTFSC           STATUS,C
                GOTO            UOK68A          
                ADDWF           REMB0, F
                BCF             STATUS,C
UOK68A          RLF             AARGB0, F

                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU1608A

                CLRF            TEMP1

                MOVLW           H'08'
                MOVWF           LOOPCOUNT

LOOPU1608B      RLF             AARGB1,W
                RLF             REMB0, F
                RLF             TEMP1, F
                MOVF            BARGB0,W
                SUBWF           REMB0, F
                CLRF            AARGB3
                CLRW
                BTFSS           STATUS,C
                INCFSZ          AARGB3,W
                SUBWF           TEMP1, F

                BTFSC           STATUS,C
                GOTO            UOK68B          
                MOVF            BARGB0,W
                ADDWF           REMB0, F
                CLRF            AARGB3
                CLRW
                BTFSC           STATUS,C
                INCFSZ          AARGB3,W
                ADDWF           TEMP1, F

                BCF             STATUS,C
UOK68B          RLF             AARGB1, F

                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU1608B
				return

; 24/16 Bit Unsigned Fixed Point Divide 

;       Input:  24 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point divisor in BARGB0, BARGB1

;       Use:    CALL    FXD2416U

;       Output: 24 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point remainder in REMB0, REMB1

;       Result: AARG, REM  <--  AARG / BARG


FXD2416U        CLRF            REMB0
                CLRF            REMB1
                CLRF            TEMPD
                RLF             AARGB0,W
                RLF             REMB1, F
                MOVF            BARGB1,W
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                RLF             AARGB0, F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416A      RLF             AARGB0,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46LA
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                GOTO            UOK46LA
UADD46LA        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46LA 		RLF             AARGB0, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416A
                RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46L8
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46L8
UADD46L8        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46L8         RLF             AARGB1, F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416B      RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46LB
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                GOTO            UOK46LB
UADD46LB        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46LB         RLF             AARGB1, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416B
                RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46L16
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                GOTO            UOK46L16
UADD46L16       ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46L16        RLF             AARGB2, F
                MOVLW           H'7'
                MOVWF           LOOPCOUNT
LOOPU2416C      RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB2,0
                GOTO            UADD46LC
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           H'1'
                SUBWF           TEMPD, F
                GOTO            UOK46LC
UADD46LC        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           H'1'
                ADDWF           TEMPD, F
UOK46LC 		RLF             AARGB2, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416C
                BTFSC           AARGB2,0
                GOTO            UOK46L
                MOVF            BARGB1,W
	        	ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
UOK46L			RETURN
	
; multiply
;
;       Input:  fixed point arguments in AARG and BARG
;
;       Output: product AARGxBARG in AARG
;

;       16x16 Bit Unsigned Fixed Point Multiply 

;       Input:  16 bit unsigned fixed point multiplicand in AARGB0
;               16 bit unsigned fixed point multiplier in BARGB0

;       Use:    CALL    FXM1616U

;       Output: 32 bit unsigned fixed point product in AARGB0



FXM1616U        CLRF    AARGB2          ; clear partial product
                CLRF    AARGB3
                MOVF    AARGB0,W
                MOVWF   TEMPB0
                MOVF    AARGB1,W
                MOVWF   TEMPB1
                MOVLW   H'08'
                MOVWF   LOOPCOUNT
LOOPUM1616A     RRF     BARGB1, F
                BTFSC   STATUS,C
                GOTO    ALUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616A
                MOVWF   LOOPCOUNT
LOOPUM1616B     RRF     BARGB0, F
                BTFSC   STATUS,C
                GOTO    BLUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616B
                CLRF    AARGB0
                CLRF    AARGB1
                RETLW   H'00'
BLUM1616NAP     BCF     STATUS,C
                GOTO    BLUM1616NA
ALUM1616NAP     BCF     STATUS,C
                GOTO    ALUM1616NA
ALOOPUM1616     RRF     BARGB1, F
                BTFSS   STATUS,C
                GOTO    ALUM1616NA
                MOVF    TEMPB1,W
                ADDWF   AARGB1, F
                MOVF    TEMPB0,W
                BTFSC   STATUS,C
                INCFSZ  TEMPB0,W
                ADDWF   AARGB0, F
ALUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                DECFSZ LOOPCOUNT, F
                GOTO   ALOOPUM1616
                MOVLW  H'08'
                MOVWF  LOOPCOUNT
BLOOPUM1616     RRF    BARGB0, F
                BTFSS  STATUS,C
                GOTO   BLUM1616NA
                MOVF   TEMPB1,W
                ADDWF  AARGB1, F
                MOVF   TEMPB0,W
                BTFSC  STATUS,C
                INCFSZ TEMPB0,W
                ADDWF  AARGB0, F
BLUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                RRF    AARGB3, F
                DECFSZ LOOPCOUNT, F
                GOTO   BLOOPUM1616
                RETURN


	end